﻿///
/// 日時データの内部表現クラスのヘッダファイル。
///

#if       !defined(INCLUDED_SZ_TIME)
#define            INCLUDED_SZ_TIME

#if       !defined(INCLUDED_SZ_COMMON)
#include  <szCommon.hpp>
#endif // !defined(INCLUDED_SZ_COMMON)

#if       !defined(INCLUDED_STD_CTIME)
#define            INCLUDED_STD_CTIME
#include  <ctime>
#endif // !defined(INCLUDED_STD_CTIME)

SZ_NS_BEG(szpp)

/// <summary>
/// Windows の FILETIME 互換の日時データクラス。
/// 
/// 唯一のデータメンバである 64 ビット整数値は 1601/1/1 00:00:00 からの経過時間を 100 ナノ秒単位で表現する。
/// </summary>
/// <notes>
/// 64 ビット整数値が唯一のデータメンバなので sizeof(Time) は 8 バイトになるはず。
/// </notes>

class SZ_SPEC Time
{
public:
  /// デフォルトコンストラクタ。
  Time() : t(0) { }
  /// コピーコンストラクタ。
  Time(const Time & rhs) : t(rhs.t) { }
  /// 64 ビット整数値からの変換コンストラクタ。
  explicit Time(s64 rhs) : t(rhs) { }

  /// 代入演算子。
  Time & operator=(const Time & rhs) { t = rhs.t; return *this; }
  /// 64 ビット整数値からの代入演算子。
  Time & operator=(const s64 & rhs) { t = rhs; return *this; }

  /// <summary>
  /// 4713BC 1/1 12:00 を基準とする標準ジュリアン日数を計算するメソッド。
  /// </summary>
  /// <remarks>
  /// Time クラスの内部処理としては JulianMidnight メソッドにより 4713BC 1/1 00:00
  /// を基準とした非標準的なジュリアン日数を計算してから加工している。
  /// </remarks>
  int Julian() const;

  /// カレンダー日時に変換するメソッド。
  void Extract(std::tm * tmbuf) const;

  /// カレンダー日時の年要素を返すメソッド。
  int Year() const;

  /// カレンダー日時の月要素を返すメソッド。一月が 1、二月が 2、…となる。
  int Month() const;

  /// カレンダー日時の日要素を返すメソッド。
  int Day() const;

  /// カレンダー日時の一年における日要素を返すメソッド。1/1 が 1、1/2 が 2、…となる。
  int DayOfYear() const;

  /// カレンダー日時の曜日要素を返すメソッド。日曜日が 0、月曜日が 1、…となる。
  int DayOfWeek() const;

  /// カレンダー日時の時要素を返すメソッド。
  int Hour() const;

  /// カレンダー日時の分要素を返すメソッド。
  int Minute() const;

  /// カレンダー日時の秒要素を返すメソッド。
  int Second() const;

  /// カレンダー日時の 100 ナノ秒要素を返すメソッド。
  int Nanosec100() const;

  /// カレンダー日時から Time インスタンスを作成する静的メソッド。
  static Time Compose(int y, int mon, int d, int h, int m, int s, int ns100);

  /// 現在の日時から Time インスタンスを作成する静的メソッド。
  static Time Now();

  /// 加算代入演算子。
  Time & operator+=(const Time & rhs) { t += rhs.t; return *this; }
  /// 減算代入演算子。
  Time & operator-=(const Time & rhs) { t -= rhs.t; return *this; }
  /// 64 ビット整数値からの加算代入演算子。
  Time & operator+=(const s64 & rhs) { t += rhs; return *this; }
  /// 64 ビット整数値からの減算代入演算子。
  Time & operator-=(const s64 & rhs) { t -= rhs; return *this; }

  /// 64 ビット整数値への変換演算子。
  operator s64() const { return t; }
  /// 64 ビット整数値を取得するメソッド。
  s64 get() const { return t; }

private:

  /// <summary>
  /// 4713BC 1/1 00:00 を基準とした非標準的なジュリアン日数を計算するメソッド。
  /// </summary>
  /// <remarks>
  /// Time クラスの実装の基本となるメソッド。
  /// </remarks>
  int JulianMidnight() const;

  /// <summary>
  /// カレンダー日 yyyy/mm/dd をジュリアン日数から計算するメソッド。
  /// </summary>
  /// <details>
  /// <para>
  /// 引数 j が Julian メソッドが返した値ならば、時刻が午後なら正しい結果となる。
  /// </para>
  /// <para>
  /// Time クラスは内部的には JulianMidnight メソッドを呼びだし、その結果を引数 j に与える。
  /// これにより、時刻によらずに、結果は常に正しいものとなる。<p>
  /// </para>
  /// <para>
  /// このためにオーバーロードされた <link szpp::Time::Mdy@int *@int *@int *@const, Mdy>
  /// が用意されており、これは内部で JulianMidnight を呼びだすようになっている。
  /// </para>
  /// </details>
  /// <remarks>
  /// この Tantzen によって考案されたカレンダー日の計算は日時が午後でなければ正しい結果を返す。
  /// 例えば、1/1 13:00 と 1/2 11:00 のどちらも 1/1 にマップされる。
  /// この障害を克服するため、便宜上 JulianMidnight メソッドを利用することが多い。
  /// </remarks>
  /// <notes>
  /// アルゴリズムの詳細については次の論文を参照 :
  /// Robert G. Tantzen,
  /// "Conversions between calendar %date and %Julian %day number.", Algorithm 199,
  /// Communications of the ACM, vol.6, no.8, Aug 1963, pp.444
  /// </notes>
  void Mdy(int j, int * m, int * d, int * y) const;

  /// カレンダー日 yyyy/mm/dd を夜中を基準とした非標準的なジュリアン日数から計算するメソッド。
  void Mdy(int * m, int * d, int * y) const;

  /// <summary>
  /// カレンダー日からジュリアン日数を計算するメソッド。
  /// </summary>
  /// <details>
  /// 本メソッドの計算結果は時刻によらず常に正しい。
  /// つまり、結果は、JulianMidnight メソッドが返すものと同じく、夜中を基準とした非標準的なジュリアン日数である。
  /// </details>
  static int Jday(int m, int d, int y);

  /// ジュリアン日数と時刻要素を組み合わせて Time インスタンスを生成する静的メソッド。
  static s64 ComposeFromJulian(int jd, int h, int m, int s, int ns100);

  /// 比較演算子。
  friend bool operator==(const Time & lhs, const Time & rhs);
  /// 比較演算子。
  friend bool operator<(const Time & lhs, const Time & rhs);

  /// 1601/01/01 00:00:00 からの経過時間を 100 ナノ秒単位で表したもの。
  /// 言い換えると、Windows の FILETIME と同じ意味合いを持つ数値。
  s64 t;
};

inline Time operator+(const Time & lhs, const Time & rhs)
{
  Time retval(lhs);
  retval += rhs;
  return retval;
}

inline Time operator-(const Time & lhs, const Time & rhs)
{
  Time retval(lhs);
  retval -= rhs;
  return retval;
}

inline Time operator+(const Time & lhs, const s64 & rhs)
{
  Time retval(lhs);
  retval += rhs;
  return retval;
}

inline Time operator-(const Time & lhs, const s64 & rhs)
{
  Time retval(lhs);
  retval -= rhs;
  return retval;
}

inline bool operator==(const Time & lhs, const Time & rhs)
{
  return (lhs.t == rhs.t);
}

inline bool operator!=(const Time & lhs, const Time & rhs)
{
  return !(lhs == rhs);
}

inline bool operator<(const Time & lhs, const Time & rhs)
{
  return (lhs.t < rhs.t);
}

inline bool operator>(const Time & lhs, const Time & rhs)
{
  return rhs < lhs;
}

inline bool operator>=(const Time & lhs, const Time & rhs)
{
  return !(lhs < rhs);
}

inline bool operator<=(const Time & lhs, const Time & rhs)
{
  return !(rhs < lhs);
}

BOOST_STATIC_ASSERT(sizeof(Time) == 8);

SZ_NS_END(szpp)

#endif // !defined(INCLUDED_SZ_TIME)
